#!/usr/bin/env python3
"""
AETHER-MESH CONTRIBUTOR NODE
Unlicensed app for mesh contributors with:
- Hardware locking to contributor's machine
- Separate metering for Storage vs CPU/GPU
- AI-powered file importance scoring
- Vault system for protected storage
- Slashing for bad behavior
- Visual particle health dashboard
- SHARD STORAGE for distributed file redundancy
"""

from flask import Flask, render_template_string, jsonify, request
import hashlib
import uuid
import json
import os
import time
import random
import threading
from datetime import datetime, timedelta

app = Flask(__name__)

# Contributor data
CONTRIBUTOR_DATA = {
    "hardware_id": None,
    "partner_id": None,
    "contributor_type": "storage",  # "storage" or "compute"
    "registered": False,
    "earnings": 0.0,
    "slashing_deductions": 0.0,
    "vault_files": [],
    "cached_files": [],
    "stored_shards": {},  # shard_id -> shard_data
    "shard_proofs": {},   # shard_id -> last_proof_time
}

# Metering data - separate for storage vs compute
STORAGE_METRICS = {
    "total_storage_gb": 100,
    "used_storage_gb": 0,
    "bandwidth_served_gb": 0,
    "files_served": 0,
    "shards_stored": 0,
    "shard_retrievals": 0,
    "uptime_hours": 0,
    "uptime_start": None,
}

COMPUTE_METRICS = {
    "cpu_cores_available": 4,
    "gpu_available": False,
    "gpu_model": None,
    "cpu_hours_contributed": 0,
    "gpu_hours_contributed": 0,
    "tasks_completed": 0,
    "tasks_failed": 0,
    "current_utilization": 0,
}

# Slashing events
SLASHING_LOG = []

# AI Cache scoring keywords
HIGH_IMPORTANCE_KEYWORDS = [
    'contract', 'legal', 'tax', 'invoice', 'receipt', 'medical', 'insurance',
    'passport', 'license', 'certificate', 'agreement', 'deed', 'will', 'trust',
    'financial', 'bank', 'statement', 'report', 'official', 'signed', 'notarized'
]

LOW_IMPORTANCE_KEYWORDS = [
    'temp', 'tmp', 'cache', 'thumb', 'preview', 'draft', 'copy', 'backup_old',
    'untitled', 'new_file', 'screenshot', 'download', 'random'
]

def get_hardware_fingerprint():
    """Generate unique hardware fingerprint"""
    components = []
    
    # Get MAC addresses
    try:
        with open('/sys/class/net/eth0/address', 'r') as f:
            components.append(f.read().strip())
    except:
        components.append(hex(uuid.getnode()))
    
    # Get CPU info
    try:
        with open('/proc/cpuinfo', 'r') as f:
            for line in f:
                if 'Serial' in line or 'model name' in line:
                    components.append(line.strip())
                    break
    except:
        components.append("cpu-fallback")
    
    # Get machine-id
    try:
        with open('/etc/machine-id', 'r') as f:
            components.append(f.read().strip())
    except:
        components.append(str(uuid.uuid4()))
    
    combined = '|'.join(components)
    return hashlib.sha256(combined.encode()).hexdigest()[:32]

def calculate_file_importance(filename, file_size, access_count=1, last_access=None, unique_accessors=1):
    """AI-powered file importance scoring (0-100)"""
    score = 50  # Base score
    filename_lower = filename.lower()
    
    # Keyword analysis
    for keyword in HIGH_IMPORTANCE_KEYWORDS:
        if keyword in filename_lower:
            score += 10
    
    for keyword in LOW_IMPORTANCE_KEYWORDS:
        if keyword in filename_lower:
            score -= 15
    
    # File extension scoring
    ext = filename.split('.')[-1].lower() if '.' in filename else ''
    ext_scores = {
        'pdf': 75, 'doc': 70, 'docx': 70, 'xls': 70, 'xlsx': 70,
        'jpg': 50, 'png': 50, 'mp4': 45, 'mp3': 40,
        'txt': 55, 'csv': 65, 'json': 60,
        'tmp': 10, 'log': 20, 'cache': 5, 'bak': 25
    }
    if ext in ext_scores:
        score = (score + ext_scores[ext]) // 2
    
    # Biometric scoring - access patterns
    if access_count > 10:
        score += 20
    elif access_count > 5:
        score += 10
    elif access_count > 2:
        score += 5
    
    # Recency bonus
    if last_access:
        days_ago = (datetime.now() - last_access).days
        if days_ago < 7:
            score += 15
        elif days_ago < 30:
            score += 8
    
    # Multi-user bonus
    if unique_accessors > 3:
        score += 15
    elif unique_accessors > 1:
        score += 8
    
    return max(0, min(100, score))

def get_retention_days(importance_score):
    """Get retention period based on importance"""
    if importance_score >= 80:
        return 365
    elif importance_score >= 60:
        return 90
    elif importance_score >= 40:
        return 30
    elif importance_score >= 20:
        return 14
    else:
        return 7

def calculate_storage_earnings():
    """Calculate earnings for storage contributors"""
    base_rate_per_gb_month = 0.01  # $0.01 per GB stored per month
    bandwidth_rate = 0.005  # $0.005 per GB served
    uptime_bonus = 0.001  # $0.001 per hour of uptime
    shard_bonus = 0.0001  # $0.0001 per shard stored
    
    storage_earnings = STORAGE_METRICS["used_storage_gb"] * base_rate_per_gb_month / 30
    bandwidth_earnings = STORAGE_METRICS["bandwidth_served_gb"] * bandwidth_rate
    uptime_earnings = STORAGE_METRICS["uptime_hours"] * uptime_bonus
    shard_earnings = STORAGE_METRICS["shards_stored"] * shard_bonus
    
    return storage_earnings + bandwidth_earnings + uptime_earnings + shard_earnings

def calculate_compute_earnings():
    """Calculate earnings for compute contributors"""
    cpu_rate_per_hour = 0.02  # $0.02 per CPU hour
    gpu_rate_per_hour = 0.15  # $0.15 per GPU hour
    task_bonus = 0.001  # $0.001 per completed task
    
    cpu_earnings = COMPUTE_METRICS["cpu_hours_contributed"] * cpu_rate_per_hour
    gpu_earnings = COMPUTE_METRICS["gpu_hours_contributed"] * gpu_rate_per_hour
    task_earnings = COMPUTE_METRICS["tasks_completed"] * task_bonus
    
    return cpu_earnings + gpu_earnings + task_earnings

def apply_slashing(reason, amount):
    """Apply slashing penalty"""
    CONTRIBUTOR_DATA["slashing_deductions"] += amount
    SLASHING_LOG.append({
        "timestamp": datetime.now().isoformat(),
        "reason": reason,
        "amount": amount
    })

def calculate_health():
    """Calculate overall node health (0-100)"""
    health = 100
    
    if CONTRIBUTOR_DATA["contributor_type"] == "storage":
        # Storage health factors
        storage_usage = STORAGE_METRICS["used_storage_gb"] / max(STORAGE_METRICS["total_storage_gb"], 1)
        if storage_usage > 0.95:
            health -= 30
        elif storage_usage > 0.85:
            health -= 15
        
        # Uptime factor
        if STORAGE_METRICS["uptime_hours"] < 24:
            health -= 20
        
        # Shard health
        shard_count = STORAGE_METRICS["shards_stored"]
        if shard_count > 100:
            health += 10
        
    else:
        # Compute health factors
        if COMPUTE_METRICS["tasks_failed"] > COMPUTE_METRICS["tasks_completed"] * 0.1:
            health -= 25
        
        utilization = COMPUTE_METRICS["current_utilization"]
        if utilization > 95:
            health -= 10
    
    # Slashing impact
    if CONTRIBUTOR_DATA["slashing_deductions"] > 0:
        health -= min(30, CONTRIBUTOR_DATA["slashing_deductions"] * 10)
    
    return max(0, min(100, health))

def calculate_shard_health():
    """Calculate health of stored shards"""
    if not CONTRIBUTOR_DATA["stored_shards"]:
        return 100
    
    healthy = 0
    total = len(CONTRIBUTOR_DATA["stored_shards"])
    
    for shard_id, shard in CONTRIBUTOR_DATA["stored_shards"].items():
        # Check if shard is valid
        if shard.get("verified", False):
            healthy += 1
    
    return int((healthy / total) * 100) if total > 0 else 100

# Shard Management Functions
def store_shard(shard_id, shard_data, file_id, shard_index, total_shards):
    """Store a file shard"""
    CONTRIBUTOR_DATA["stored_shards"][shard_id] = {
        "data": shard_data,
        "file_id": file_id,
        "shard_index": shard_index,
        "total_shards": total_shards,
        "stored_at": datetime.now().isoformat(),
        "verified": True,
        "size_bytes": len(shard_data) if isinstance(shard_data, bytes) else len(shard_data.encode())
    }
    STORAGE_METRICS["shards_stored"] = len(CONTRIBUTOR_DATA["stored_shards"])
    return True

def retrieve_shard(shard_id):
    """Retrieve a stored shard"""
    if shard_id in CONTRIBUTOR_DATA["stored_shards"]:
        STORAGE_METRICS["shard_retrievals"] += 1
        return CONTRIBUTOR_DATA["stored_shards"][shard_id]["data"]
    return None

def prove_shard(shard_id):
    """Generate proof of shard possession"""
    if shard_id in CONTRIBUTOR_DATA["stored_shards"]:
        shard = CONTRIBUTOR_DATA["stored_shards"][shard_id]
        proof = hashlib.sha256(str(shard["data"]).encode()).hexdigest()
        CONTRIBUTOR_DATA["shard_proofs"][shard_id] = datetime.now().isoformat()
        return proof
    return None

DASHBOARD_HTML = '''
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AETHER-MESH Contributor Node</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            background: #0a0a0f;
            color: #e0e0e0;
            font-family: 'Segoe UI', system-ui, sans-serif;
            min-height: 100vh;
            overflow-x: hidden;
        }
        
        #particle-canvas {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 0;
        }
        
        .container {
            position: relative;
            z-index: 1;
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            text-align: center;
            padding: 30px 0;
            border-bottom: 1px solid rgba(0, 255, 136, 0.2);
            margin-bottom: 30px;
        }
        
        .header h1 {
            font-size: 2.5rem;
            background: linear-gradient(135deg, #00ff88, #00d4ff);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin-bottom: 10px;
        }
        
        .node-type {
            display: inline-block;
            padding: 8px 20px;
            background: linear-gradient(135deg, rgba(0, 255, 136, 0.2), rgba(0, 212, 255, 0.2));
            border: 1px solid #00ff88;
            border-radius: 20px;
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 2px;
        }
        
        .health-ring {
            width: 200px;
            height: 200px;
            margin: 30px auto;
            position: relative;
        }
        
        .health-ring svg {
            transform: rotate(-90deg);
        }
        
        .health-ring .bg {
            fill: none;
            stroke: rgba(255, 255, 255, 0.1);
            stroke-width: 12;
        }
        
        .health-ring .progress {
            fill: none;
            stroke-width: 12;
            stroke-linecap: round;
            transition: stroke-dashoffset 1s ease, stroke 0.5s ease;
        }
        
        .health-value {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            text-align: center;
        }
        
        .health-value .number {
            font-size: 3rem;
            font-weight: bold;
        }
        
        .health-value .label {
            font-size: 0.9rem;
            opacity: 0.7;
        }
        
        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 30px;
        }
        
        .card {
            background: rgba(20, 20, 30, 0.8);
            border: 1px solid rgba(0, 255, 136, 0.2);
            border-radius: 15px;
            padding: 25px;
            backdrop-filter: blur(10px);
        }
        
        .card h3 {
            color: #00ff88;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .stat-row {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.05);
        }
        
        .stat-row:last-child {
            border-bottom: none;
        }
        
        .stat-label {
            opacity: 0.7;
        }
        
        .stat-value {
            font-weight: bold;
            color: #00d4ff;
        }
        
        .stat-value.earnings {
            color: #00ff88;
        }
        
        .stat-value.penalty {
            color: #ff4444;
        }
        
        .progress-bar {
            height: 8px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 4px;
            overflow: hidden;
            margin-top: 8px;
        }
        
        .progress-bar .fill {
            height: 100%;
            border-radius: 4px;
            transition: width 0.5s ease;
        }
        
        .shard-grid {
            display: grid;
            grid-template-columns: repeat(10, 1fr);
            gap: 4px;
            margin-top: 15px;
        }
        
        .shard-cell {
            aspect-ratio: 1;
            border-radius: 3px;
            transition: all 0.3s ease;
        }
        
        .shard-cell.healthy {
            background: #00ff88;
            box-shadow: 0 0 10px rgba(0, 255, 136, 0.5);
        }
        
        .shard-cell.warning {
            background: #ffaa00;
            box-shadow: 0 0 10px rgba(255, 170, 0, 0.5);
        }
        
        .shard-cell.empty {
            background: rgba(255, 255, 255, 0.1);
        }
        
        .vault-list {
            max-height: 200px;
            overflow-y: auto;
        }
        
        .vault-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px;
            background: rgba(0, 0, 0, 0.3);
            border-radius: 8px;
            margin-bottom: 8px;
        }
        
        .vault-item .icon {
            color: #ffd700;
        }
        
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: bold;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #00ff88, #00d4ff);
            color: #000;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(0, 255, 136, 0.4);
        }
        
        .status-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 0.8rem;
            font-weight: bold;
        }
        
        .status-badge.online {
            background: rgba(0, 255, 136, 0.2);
            color: #00ff88;
        }
        
        .status-badge.offline {
            background: rgba(255, 68, 68, 0.2);
            color: #ff4444;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .live-indicator {
            display: inline-block;
            width: 8px;
            height: 8px;
            background: #00ff88;
            border-radius: 50%;
            margin-right: 8px;
            animation: pulse 2s infinite;
        }
    </style>
</head>
<body>
    <canvas id="particle-canvas"></canvas>
    
    <div class="container">
        <div class="header">
            <h1>⚡ AETHER-MESH CONTRIBUTOR</h1>
            <div class="node-type" id="node-type">{{ contributor_type | upper }} NODE</div>
            <p style="margin-top: 15px; opacity: 0.7;">
                <span class="live-indicator"></span>
                Hardware ID: {{ hardware_id[:16] }}...
            </p>
        </div>
        
        <div class="health-ring">
            <svg width="200" height="200" viewBox="0 0 200 200">
                <circle class="bg" cx="100" cy="100" r="85"></circle>
                <circle class="progress" id="health-progress" cx="100" cy="100" r="85" 
                        stroke-dasharray="534" stroke-dashoffset="534"></circle>
            </svg>
            <div class="health-value">
                <div class="number" id="health-number">{{ health }}%</div>
                <div class="label">NODE HEALTH</div>
            </div>
        </div>
        
        <div class="grid">
            <!-- Earnings Card -->
            <div class="card">
                <h3>💰 Earnings</h3>
                <div class="stat-row">
                    <span class="stat-label">Gross Earnings</span>
                    <span class="stat-value earnings">${{ "%.4f"|format(gross_earnings) }}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Slashing Deductions</span>
                    <span class="stat-value penalty">-${{ "%.4f"|format(slashing) }}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Net Earnings</span>
                    <span class="stat-value earnings" style="font-size: 1.3rem;">${{ "%.4f"|format(net_earnings) }}</span>
                </div>
                <button class="btn btn-primary" style="width: 100%; margin-top: 20px;">
                    Request Payout
                </button>
            </div>
            
            {% if contributor_type == 'storage' %}
            <!-- Storage Metrics -->
            <div class="card">
                <h3>💾 Storage Metrics</h3>
                <div class="stat-row">
                    <span class="stat-label">Storage Used</span>
                    <span class="stat-value">{{ storage_metrics.used_storage_gb }} / {{ storage_metrics.total_storage_gb }} GB</span>
                </div>
                <div class="progress-bar">
                    <div class="fill" style="width: {{ (storage_metrics.used_storage_gb / storage_metrics.total_storage_gb * 100)|int }}%; background: linear-gradient(90deg, #00ff88, #00d4ff);"></div>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Bandwidth Served</span>
                    <span class="stat-value">{{ "%.2f"|format(storage_metrics.bandwidth_served_gb) }} GB</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Uptime</span>
                    <span class="stat-value">{{ storage_metrics.uptime_hours }} hours</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Files Served</span>
                    <span class="stat-value">{{ storage_metrics.files_served }}</span>
                </div>
            </div>
            
            <!-- Shard Storage -->
            <div class="card">
                <h3>🧩 Shard Storage</h3>
                <div class="stat-row">
                    <span class="stat-label">Shards Stored</span>
                    <span class="stat-value">{{ storage_metrics.shards_stored }}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Shard Retrievals</span>
                    <span class="stat-value">{{ storage_metrics.shard_retrievals }}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Shard Health</span>
                    <span class="stat-value">{{ shard_health }}%</span>
                </div>
                <div class="shard-grid" id="shard-grid">
                    {% for i in range(50) %}
                    <div class="shard-cell {% if i < storage_metrics.shards_stored %}healthy{% else %}empty{% endif %}"></div>
                    {% endfor %}
                </div>
            </div>
            
            {% else %}
            <!-- Compute Metrics -->
            <div class="card">
                <h3>🖥️ CPU Metrics</h3>
                <div class="stat-row">
                    <span class="stat-label">CPU Cores</span>
                    <span class="stat-value">{{ compute_metrics.cpu_cores_available }}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">CPU Hours</span>
                    <span class="stat-value">{{ "%.2f"|format(compute_metrics.cpu_hours_contributed) }} hrs</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Current Load</span>
                    <span class="stat-value">{{ compute_metrics.current_utilization }}%</span>
                </div>
                <div class="progress-bar">
                    <div class="fill" style="width: {{ compute_metrics.current_utilization }}%; background: linear-gradient(90deg, #00ff88, #ffaa00, #ff4444);"></div>
                </div>
            </div>
            
            <!-- GPU Metrics -->
            <div class="card">
                <h3>🎮 GPU Metrics</h3>
                {% if compute_metrics.gpu_available %}
                <div class="stat-row">
                    <span class="stat-label">GPU Model</span>
                    <span class="stat-value">{{ compute_metrics.gpu_model }}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">GPU Hours</span>
                    <span class="stat-value">{{ "%.2f"|format(compute_metrics.gpu_hours_contributed) }} hrs</span>
                </div>
                {% else %}
                <p style="opacity: 0.5; text-align: center; padding: 20px;">No GPU detected</p>
                {% endif %}
                <div class="stat-row">
                    <span class="stat-label">Tasks Completed</span>
                    <span class="stat-value">{{ compute_metrics.tasks_completed }}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Tasks Failed</span>
                    <span class="stat-value penalty">{{ compute_metrics.tasks_failed }}</span>
                </div>
            </div>
            {% endif %}
            
            <!-- Vault -->
            <div class="card">
                <h3>🔐 Vault (Protected Storage)</h3>
                <p style="opacity: 0.7; margin-bottom: 15px; font-size: 0.9rem;">
                    Files in your vault are never auto-deleted
                </p>
                <div class="vault-list">
                    {% if vault_files %}
                        {% for file in vault_files %}
                        <div class="vault-item">
                            <span><span class="icon">🔒</span> {{ file.name }}</span>
                            <span style="opacity: 0.5;">{{ file.size }}</span>
                        </div>
                        {% endfor %}
                    {% else %}
                        <p style="opacity: 0.5; text-align: center;">No vault files yet</p>
                    {% endif %}
                </div>
            </div>
            
            <!-- Slashing Log -->
            <div class="card">
                <h3>⚠️ Slashing Events</h3>
                {% if slashing_log %}
                    {% for event in slashing_log[-5:] %}
                    <div class="stat-row">
                        <span class="stat-label">{{ event.reason }}</span>
                        <span class="stat-value penalty">-${{ "%.4f"|format(event.amount) }}</span>
                    </div>
                    {% endfor %}
                {% else %}
                    <p style="opacity: 0.5; text-align: center; padding: 20px;">
                        ✅ No slashing events - keep it up!
                    </p>
                {% endif %}
            </div>
        </div>
    </div>
    
    <script>
        // Particle System based on node health
        const canvas = document.getElementById('particle-canvas');
        const ctx = canvas.getContext('2d');
        
        let health = {{ health }};
        let contributorType = '{{ contributor_type }}';
        let shardHealth = {{ shard_health }};
        
        function resizeCanvas() {
            canvas.width = window.innerWidth;
            canvas.height = window.innerHeight;
        }
        resizeCanvas();
        window.addEventListener('resize', resizeCanvas);
        
        class Particle {
            constructor() {
                this.reset();
            }
            
            reset() {
                this.x = Math.random() * canvas.width;
                this.y = Math.random() * canvas.height;
                this.size = Math.random() * 3 + 1;
                this.speedX = (Math.random() - 0.5) * 2;
                this.speedY = (Math.random() - 0.5) * 2;
                this.life = 1;
                this.decay = Math.random() * 0.01 + 0.005;
                
                // Color based on health
                if (health >= 80) {
                    this.hue = 145; // Green
                } else if (health >= 50) {
                    this.hue = 60; // Yellow
                } else {
                    this.hue = 0; // Red
                }
                
                // Add cyan particles for storage/shards
                if (contributorType === 'storage' && Math.random() > 0.5) {
                    this.hue = 180; // Cyan for storage
                }
            }
            
            update() {
                this.x += this.speedX;
                this.y += this.speedY;
                this.life -= this.decay;
                
                if (this.life <= 0 || this.x < 0 || this.x > canvas.width || 
                    this.y < 0 || this.y > canvas.height) {
                    this.reset();
                }
            }
            
            draw() {
                ctx.beginPath();
                ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2);
                ctx.fillStyle = `hsla(${this.hue}, 100%, 60%, ${this.life * 0.6})`;
                ctx.fill();
                
                // Glow effect
                ctx.shadowBlur = 15;
                ctx.shadowColor = `hsla(${this.hue}, 100%, 60%, ${this.life * 0.3})`;
            }
        }
        
        // Number of particles based on health
        const particleCount = Math.floor(50 + (health * 1.5));
        const particles = Array.from({ length: particleCount }, () => new Particle());
        
        function animate() {
            ctx.fillStyle = 'rgba(10, 10, 15, 0.1)';
            ctx.fillRect(0, 0, canvas.width, canvas.height);
            
            particles.forEach(p => {
                p.update();
                p.draw();
            });
            
            // Draw connections between close particles
            ctx.shadowBlur = 0;
            for (let i = 0; i < particles.length; i++) {
                for (let j = i + 1; j < particles.length; j++) {
                    const dx = particles[i].x - particles[j].x;
                    const dy = particles[i].y - particles[j].y;
                    const dist = Math.sqrt(dx * dx + dy * dy);
                    
                    if (dist < 100) {
                        ctx.beginPath();
                        ctx.moveTo(particles[i].x, particles[i].y);
                        ctx.lineTo(particles[j].x, particles[j].y);
                        ctx.strokeStyle = `rgba(0, 255, 136, ${(1 - dist / 100) * 0.2})`;
                        ctx.stroke();
                    }
                }
            }
            
            requestAnimationFrame(animate);
        }
        animate();
        
        // Update health ring
        const healthProgress = document.getElementById('health-progress');
        const circumference = 2 * Math.PI * 85;
        const offset = circumference - (health / 100) * circumference;
        healthProgress.style.strokeDashoffset = offset;
        
        // Color based on health
        if (health >= 80) {
            healthProgress.style.stroke = '#00ff88';
        } else if (health >= 50) {
            healthProgress.style.stroke = '#ffaa00';
        } else {
            healthProgress.style.stroke = '#ff4444';
        }
        
        // Auto-refresh data
        setInterval(() => {
            fetch('/api/status')
                .then(r => r.json())
                .then(data => {
                    document.getElementById('health-number').textContent = data.health + '%';
                    // Update other values...
                });
        }, 5000);
    </script>
</body>
</html>
'''

@app.route('/')
def dashboard():
    if CONTRIBUTOR_DATA["contributor_type"] == "storage":
        gross = calculate_storage_earnings()
    else:
        gross = calculate_compute_earnings()
    
    net = gross - CONTRIBUTOR_DATA["slashing_deductions"]
    
    return render_template_string(DASHBOARD_HTML,
        hardware_id=CONTRIBUTOR_DATA["hardware_id"] or get_hardware_fingerprint(),
        contributor_type=CONTRIBUTOR_DATA["contributor_type"],
        health=calculate_health(),
        shard_health=calculate_shard_health(),
        gross_earnings=gross,
        slashing=CONTRIBUTOR_DATA["slashing_deductions"],
        net_earnings=max(0, net),
        storage_metrics=STORAGE_METRICS,
        compute_metrics=COMPUTE_METRICS,
        vault_files=CONTRIBUTOR_DATA["vault_files"],
        slashing_log=SLASHING_LOG
    )

@app.route('/api/status')
def api_status():
    if CONTRIBUTOR_DATA["contributor_type"] == "storage":
        gross = calculate_storage_earnings()
    else:
        gross = calculate_compute_earnings()
    
    return jsonify({
        "hardware_id": CONTRIBUTOR_DATA["hardware_id"],
        "contributor_type": CONTRIBUTOR_DATA["contributor_type"],
        "health": calculate_health(),
        "shard_health": calculate_shard_health(),
        "gross_earnings": gross,
        "slashing_deductions": CONTRIBUTOR_DATA["slashing_deductions"],
        "net_earnings": max(0, gross - CONTRIBUTOR_DATA["slashing_deductions"]),
        "storage_metrics": STORAGE_METRICS,
        "compute_metrics": COMPUTE_METRICS,
        "shards_stored": len(CONTRIBUTOR_DATA["stored_shards"]),
        "online": True
    })

@app.route('/api/register', methods=['POST'])
def register():
    """Register with a partner's network"""
    data = request.json
    CONTRIBUTOR_DATA["partner_id"] = data.get("partner_id")
    CONTRIBUTOR_DATA["hardware_id"] = get_hardware_fingerprint()
    CONTRIBUTOR_DATA["contributor_type"] = data.get("type", "storage")
    CONTRIBUTOR_DATA["registered"] = True
    
    if STORAGE_METRICS["uptime_start"] is None:
        STORAGE_METRICS["uptime_start"] = datetime.now()
    
    return jsonify({
        "success": True,
        "hardware_id": CONTRIBUTOR_DATA["hardware_id"],
        "message": "Registered with partner network"
    })

@app.route('/api/shard/store', methods=['POST'])
def api_store_shard():
    """Store a file shard"""
    data = request.json
    shard_id = data.get("shard_id")
    shard_data = data.get("data")
    file_id = data.get("file_id")
    shard_index = data.get("shard_index", 0)
    total_shards = data.get("total_shards", 1)
    
    success = store_shard(shard_id, shard_data, file_id, shard_index, total_shards)
    
    return jsonify({
        "success": success,
        "shard_id": shard_id,
        "stored_at": CONTRIBUTOR_DATA["hardware_id"]
    })

@app.route('/api/shard/retrieve/<shard_id>')
def api_retrieve_shard(shard_id):
    """Retrieve a stored shard"""
    data = retrieve_shard(shard_id)
    if data:
        return jsonify({"success": True, "data": data})
    return jsonify({"success": False, "error": "Shard not found"}), 404

@app.route('/api/shard/prove/<shard_id>')
def api_prove_shard(shard_id):
    """Prove possession of a shard"""
    proof = prove_shard(shard_id)
    if proof:
        return jsonify({"success": True, "proof": proof})
    return jsonify({"success": False, "error": "Shard not found"}), 404

@app.route('/api/vault/add', methods=['POST'])
def add_to_vault():
    """Add file to protected vault"""
    data = request.json
    CONTRIBUTOR_DATA["vault_files"].append({
        "name": data.get("name"),
        "size": data.get("size", "Unknown"),
        "added": datetime.now().isoformat()
    })
    return jsonify({"success": True})

@app.route('/api/compute/task', methods=['POST'])
def receive_compute_task():
    """Receive a compute task (for compute contributors)"""
    if CONTRIBUTOR_DATA["contributor_type"] != "compute":
        return jsonify({"error": "Not a compute node"}), 400
    
    data = request.json
    # Simulate task processing
    COMPUTE_METRICS["tasks_completed"] += 1
    COMPUTE_METRICS["cpu_hours_contributed"] += data.get("cpu_hours", 0.1)
    if COMPUTE_METRICS["gpu_available"]:
        COMPUTE_METRICS["gpu_hours_contributed"] += data.get("gpu_hours", 0)
    
    return jsonify({"success": True, "task_id": str(uuid.uuid4())})

if __name__ == '__main__':
    CONTRIBUTOR_DATA["hardware_id"] = get_hardware_fingerprint()
    print(f"🚀 Contributor Node starting...")
    print(f"🔐 Hardware ID: {CONTRIBUTOR_DATA['hardware_id']}")
    
    # Start uptime tracking
    STORAGE_METRICS["uptime_start"] = datetime.now()
    
    # Simulate some initial data for demo
    STORAGE_METRICS["used_storage_gb"] = 23
    STORAGE_METRICS["bandwidth_served_gb"] = 156.7
    STORAGE_METRICS["uptime_hours"] = 482
    STORAGE_METRICS["files_served"] = 1247
    
    # Add some demo shards
    for i in range(15):
        store_shard(f"shard_{i}", f"demo_data_{i}", f"file_{i//3}", i % 5, 5)
    
    app.run(host='0.0.0.0', port=5003, debug=True)
